<?php //phpcs:ignore WordPress.Files.FileName.InvalidClassFileName
/**
 * This file belongs to the YIT Plugin Framework.
 *
 * This source file is subject to the GNU GENERAL PUBLIC LICENSE (GPL 3.0)
 * that is bundled with this package in the file LICENSE.txt.
 * It is also available through the world-wide-web at this URL:
 * http://www.gnu.org/licenses/gpl-3.0.txt
 *
 * @package YITH WooCommerce Request A Quote Premium
 */

if ( ! defined( 'ABSPATH' ) || ! defined( 'YITH_YWRAQ_VERSION' ) ) {
	exit; // Exit if accessed directly.
}

/**
 * Implements the YITH_YWRAQ_Cron class.
 *
 * @class    YITH_YWRAQ_Cron
 * @package  YITH
 * @since    1.4.9
 * @author   YITH
 */
class YITH_YWRAQ_Cron {

	/**
	 * Single instance of the class
	 *
	 * @var \YITH_YWRAQ_Cron
	 */
	protected static $instance;

	/**
	 * Automatic quote
	 *
	 * @var string
	 */
	private $automatic_quote;


	/**
	 * Cron time
	 *
	 * @var int
	 */
	private $cron_time;

	/**
	 * Cron time type
	 *
	 * @var string
	 */
	private $cron_time_type;


	/**
	 * Returns single instance of the class
	 *
	 * @return \YITH_YWRAQ_Cron
	 * @since 1.0.0
	 */
	public static function get_instance() {
		if ( is_null( self::$instance ) ) {
			self::$instance = new self();
		}

		return self::$instance;
	}

	/**
	 * Constructor
	 *
	 * Initialize plugin and registers actions and filters to be used
	 *
	 * @since  1.0.0
	 * @author Emanuela Castorina
	 */
	public function __construct() {

		add_action( 'wp_loaded', array( $this, 'ywraq_set_cron' ) );

		$this->automatic_quote = get_option( 'ywraq_automate_send_quote', 'no' );
		if ( 'yes' === $this->automatic_quote ) {
			$cron_option          = get_option( 'ywraq_cron_time' );
			$this->cron_time      = isset( $cron_option['time'] ) && ! empty( $cron_option['time'] ) ? (int) $cron_option['time'] : 0;
			$this->cron_time_type = isset( $cron_option['type'] ) ? $cron_option['type'] : 'hours';

			if ( 0 !== $this->cron_time ) {
				add_filter( 'cron_schedules', array( $this, 'cron_schedule' ), 50 ); //phpcs:ignore
				add_action( 'ywraq_automatic_quote', array( $this, 'send_automatic_quote' ) );
			}
		}

		add_action( 'ywraq_clean_cron', array( $this, 'clean_session' ) );
		add_action( 'ywraq_time_validation', array( $this, 'time_validation' ) );

	}

	/**
	 * Set Cron
	 */
	public function ywraq_set_cron() {

		if ( ! wp_next_scheduled( 'ywraq_time_validation' ) ) {
			$ve = get_option( 'gmt_offset' ) > 0 ? '+' : '-';
			wp_schedule_event( strtotime( '00:00 tomorrow ' . $ve . get_option( 'gmt_offset' ) . ' HOURS' ), 'daily', 'ywraq_time_validation' );
		}

		if ( ! wp_next_scheduled( 'ywraq_clean_cron' ) ) {
			wp_schedule_event( time(), 'daily', 'ywraq_clean_cron' );
		}

		if ( ! wp_next_scheduled( 'ywraq_automatic_quote' ) && 'yes' === $this->automatic_quote && 0 !== $this->cron_time ) {
			wp_schedule_event( time(), 'ywraq_gap', 'ywraq_automatic_quote' );
		}
	}

	/**
	 * Cron Schedule
	 *
	 * Add new schedules to WordPress.
	 *
	 * @param array $schedules Schedules.
	 *
	 * @return mixed
	 * @since  1.0.0
	 */
	public function cron_schedule( $schedules ) {

		$interval = 0;

		if ( 'hours' === $this->cron_time_type ) {
			$interval = 60 * 60 * $this->cron_time;
		} elseif ( 'days' === $this->cron_time_type ) {
			$interval = 24 * 60 * 60 * $this->cron_time;
		} elseif ( 'minutes' === $this->cron_time_type ) {
			$interval = 60 * $this->cron_time;
		}

		$schedules['ywraq_gap'] = array(
			'interval' => $interval,
			'display'  => esc_html__( 'YITH WooCommerce Request a Quote Cron', 'yith-woocommerce-request-a-quote' ),
		);

		return $schedules;
	}

	/**
	 * Clean the session on database
	 */
	public function clean_session() {
		global $wpdb;

		$cookie_name = '_' . ywraq_get_cookie_name() . '_%';
		$wpdb->query( $wpdb->prepare( 'DELETE FROM ' . $wpdb->prefix . 'options  WHERE option_name LIKE %s', $cookie_name ) ); //phpcs:ignore

	}

	/**
	 * Function called from Cron to swich in
	 * ywraq-expired order status the request expired
	 *
	 * @return  void
	 * @author  Emanuela Castorina
	 * @since   1.4.9
	 */
	public function time_validation() {
		// todo:replace get_posts with wc_get_orders.
		$orders = get_posts(
			array(
				'numberposts' => -1,
				'meta_query'  => array( //phpcs:ignore
					array(
						'key'     => '_ywcm_request_expire',
						'value'   => '',
						'compare' => '!=',
					),
				),
				'post_type'   => 'shop_order',
				'post_status' => array( 'wc-ywraq-pending' ),
			)
		);

		foreach ( $orders as $order ) {
			$status = $order->post_status;
			$order  = wc_get_order( $order->ID );
			$expired_data  = strtotime( get_post_meta( $order->get_id(), '_ywcm_request_expire', true ) );
			$expired_data += ( 24 * 60 * 60 ) - 1;
			do_action( 'send_reminder_quote_mail', $order->get_id(), $expired_data );
			if ( $expired_data < time() && 'wc-ywraq-pending' === $status ) {
				$order->update_status( 'ywraq-expired' );
			}
		}
	}

	/**
	 * Send automatic quote
	 *
	 * @return  void
	 * @author  Emanuela Castorina
	 * @since   1.4.9
	 */
	public function send_automatic_quote() {

		if ( 'yes' !== $this->automatic_quote ) {
			return;
		}

		$orders = wc_get_orders(
			array(
				'numberposts' => -1,
				'status'      => array( 'wc-ywraq-new' ),
			)
		);

		if ( $orders ) {
			foreach ( $orders as $order ) {
				$order_id = $order->get_id();
				do_action( 'create_pdf', $order_id );
				do_action( 'send_quote_mail', $order_id );
				$order->update_status( 'ywraq-pending' );
			}
		}

	}
}


/**
 * Unique access to instance of YITH_YWRAQ_Cron class
 *
 * @return \YITH_YWRAQ_Cron
 */
function YITH_YWRAQ_Cron() { //phpcs:ignore
	return YITH_YWRAQ_Cron::get_instance();
}

YITH_YWRAQ_Cron();
